/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.validation.policy;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.enumerations.EncryptionAlgorithm;
import eu.europa.esig.dss.enumerations.SignatureAlgorithm;
import eu.europa.esig.dss.model.policy.CryptographicSuite;
import eu.europa.esig.dss.model.policy.SignatureAlgorithmWithMinKeySize;
import eu.europa.esig.dss.model.policy.crypto.CryptographicSuiteEvaluation;
import eu.europa.esig.dss.model.policy.crypto.CryptographicSuiteParameter;
import eu.europa.esig.dss.utils.Utils;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CryptographicSuiteUtils {
    private static final Logger LOG = LoggerFactory.getLogger(CryptographicSuiteUtils.class);
    public static final String MODULES_LENGTH_PARAMETER = "moduluslength";
    public static final String PLENGTH_PARAMETER = "plength";
    public static final String QLENGTH_PARAMETER = "qlength";

    private CryptographicSuiteUtils() {
    }

    public static boolean isSignatureAlgorithmReliable(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm) {
        if (cryptographicSuite == null) {
            return true;
        }
        return signatureAlgorithm != null && cryptographicSuite.getAcceptableSignatureAlgorithms().containsKey(signatureAlgorithm);
    }

    public static boolean isDigestAlgorithmReliable(CryptographicSuite cryptographicSuite, DigestAlgorithm digestAlgorithm) {
        if (cryptographicSuite == null) {
            return true;
        }
        return digestAlgorithm != null && cryptographicSuite.getAcceptableDigestAlgorithms().containsKey(digestAlgorithm);
    }

    public static boolean isSignatureAlgorithmWithKeySizeReliable(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm, String keyLength) {
        int keySize = CryptographicSuiteUtils.parseKeySize(keyLength);
        return CryptographicSuiteUtils.isSignatureAlgorithmWithKeySizeReliable(cryptographicSuite, signatureAlgorithm, keySize);
    }

    public static boolean isSignatureAlgorithmWithKeySizeReliable(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm, Integer keySize) {
        if (cryptographicSuite == null) {
            return true;
        }
        if (signatureAlgorithm != null && keySize != null) {
            Map<SignatureAlgorithm, Set<CryptographicSuiteEvaluation>> acceptableSignatureAlgorithms = cryptographicSuite.getAcceptableSignatureAlgorithms();
            if (!acceptableSignatureAlgorithms.containsKey(signatureAlgorithm)) {
                return false;
            }
            Set<CryptographicSuiteEvaluation> evaluations = acceptableSignatureAlgorithms.get(signatureAlgorithm);
            if (Utils.isCollectionNotEmpty(evaluations)) {
                for (CryptographicSuiteEvaluation evaluation : evaluations) {
                    if (!CryptographicSuiteUtils.isEvaluationApplicable(signatureAlgorithm.getEncryptionAlgorithm(), keySize, evaluation)) continue;
                    return true;
                }
            } else {
                return true;
            }
        }
        return false;
    }

    private static int parseKeySize(String keyLength) {
        return Utils.isStringDigits(keyLength) ? Integer.parseInt(keyLength) : 0;
    }

    public static boolean isSignatureAlgorithmKeyLengthBigEnough(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm, String keyLength) {
        if (cryptographicSuite == null) {
            return false;
        }
        int keySize = CryptographicSuiteUtils.parseKeySize(keyLength);
        if (signatureAlgorithm != null) {
            Map<SignatureAlgorithm, Set<CryptographicSuiteEvaluation>> acceptableSignatureAlgorithms = cryptographicSuite.getAcceptableSignatureAlgorithms();
            if (!acceptableSignatureAlgorithms.containsKey(signatureAlgorithm)) {
                return false;
            }
            Set<CryptographicSuiteEvaluation> evaluations = acceptableSignatureAlgorithms.get(signatureAlgorithm);
            if (Utils.isCollectionNotEmpty(evaluations)) {
                for (CryptographicSuiteEvaluation evaluation : evaluations) {
                    List<CryptographicSuiteParameter> parameterList = evaluation.getParameterList();
                    if (parameterList == null || !Utils.isCollectionNotEmpty(parameterList)) continue;
                    for (CryptographicSuiteParameter parameter : parameterList) {
                        Integer parameterMin;
                        if (!CryptographicSuiteUtils.isSupported(signatureAlgorithm.getEncryptionAlgorithm(), parameter) || (parameterMin = parameter.getMin()) != null && parameterMin >= keySize) continue;
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public static Date getExpirationDate(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm, String keyLength) {
        int keySize = CryptographicSuiteUtils.parseKeySize(keyLength);
        return CryptographicSuiteUtils.getExpirationDate(cryptographicSuite, signatureAlgorithm, keySize);
    }

    public static Date getExpirationDate(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm, Integer keySize) {
        Map<SignatureAlgorithm, Set<CryptographicSuiteEvaluation>> acceptableSignatureAlgorithms;
        Set<CryptographicSuiteEvaluation> evaluations;
        if (cryptographicSuite == null) {
            return null;
        }
        Date expirationDate = null;
        if (signatureAlgorithm != null && keySize != null && Utils.isCollectionNotEmpty(evaluations = (acceptableSignatureAlgorithms = cryptographicSuite.getAcceptableSignatureAlgorithms()).get(signatureAlgorithm))) {
            for (CryptographicSuiteEvaluation evaluation : evaluations) {
                if (!CryptographicSuiteUtils.isEvaluationApplicable(signatureAlgorithm.getEncryptionAlgorithm(), keySize, evaluation)) continue;
                Date validityEnd = evaluation.getValidityEnd();
                if (validityEnd == null) {
                    return null;
                }
                if (expirationDate != null && !validityEnd.after(expirationDate)) continue;
                expirationDate = validityEnd;
            }
        }
        return expirationDate;
    }

    public static Date getExpirationDate(CryptographicSuite cryptographicSuite, DigestAlgorithm digestAlgorithm) {
        Map<DigestAlgorithm, Set<CryptographicSuiteEvaluation>> acceptableDigestAlgorithms;
        Set<CryptographicSuiteEvaluation> evaluations;
        if (cryptographicSuite == null) {
            return null;
        }
        Date expirationDate = null;
        if (digestAlgorithm != null && Utils.isCollectionNotEmpty(evaluations = (acceptableDigestAlgorithms = cryptographicSuite.getAcceptableDigestAlgorithms()).get(digestAlgorithm))) {
            for (CryptographicSuiteEvaluation evaluation : evaluations) {
                Date validityEnd = evaluation.getValidityEnd();
                if (validityEnd == null) {
                    return null;
                }
                if (expirationDate != null && !validityEnd.after(expirationDate)) continue;
                expirationDate = validityEnd;
            }
        }
        return expirationDate;
    }

    public static boolean isDigestAlgorithmReliableAtTime(CryptographicSuite cryptographicSuite, DigestAlgorithm digestAlgorithm, Date validationTime) {
        Set<CryptographicSuiteEvaluation> evaluations = cryptographicSuite.getAcceptableDigestAlgorithms().get(digestAlgorithm);
        return CryptographicSuiteUtils.reliableEvaluationExistsAtTime(evaluations, validationTime);
    }

    public static Set<DigestAlgorithm> getReliableDigestAlgorithmsAtTime(CryptographicSuite cryptographicSuite, Date validationTime) {
        HashSet<DigestAlgorithm> reliableDigestAlgorithms = new HashSet<DigestAlgorithm>();
        for (Map.Entry<DigestAlgorithm, Set<CryptographicSuiteEvaluation>> entry : cryptographicSuite.getAcceptableDigestAlgorithms().entrySet()) {
            Set<CryptographicSuiteEvaluation> evaluations = entry.getValue();
            if (!CryptographicSuiteUtils.reliableEvaluationExistsAtTime(evaluations, validationTime)) continue;
            reliableDigestAlgorithms.add(entry.getKey());
        }
        return reliableDigestAlgorithms;
    }

    private static boolean reliableEvaluationExistsAtTime(Collection<CryptographicSuiteEvaluation> evaluations, Date validationTime) {
        return CryptographicSuiteUtils.reliableEvaluationExistsAtTime(null, null, evaluations, validationTime);
    }

    private static boolean reliableEvaluationExistsAtTime(EncryptionAlgorithm encryptionAlgorithm, Integer keySize, Collection<CryptographicSuiteEvaluation> evaluations, Date validationTime) {
        if (Utils.isCollectionNotEmpty(evaluations)) {
            for (CryptographicSuiteEvaluation evaluation : evaluations) {
                if (encryptionAlgorithm != null && !CryptographicSuiteUtils.isEvaluationApplicable(encryptionAlgorithm, keySize, evaluation)) continue;
                Date validityStart = evaluation.getValidityStart();
                Date validityEnd = evaluation.getValidityEnd();
                if (validityStart != null && !validityStart.before(validationTime) || validityEnd != null && !validityEnd.after(validationTime)) continue;
                return true;
            }
        }
        return false;
    }

    public static Set<SignatureAlgorithmWithMinKeySize> getReliableSignatureAlgorithmsWithMinimalKeyLengthAtTime(CryptographicSuite cryptographicSuite, Date validationTime) {
        HashSet<SignatureAlgorithmWithMinKeySize> result = new HashSet<SignatureAlgorithmWithMinKeySize>();
        for (Map.Entry<SignatureAlgorithm, Set<CryptographicSuiteEvaluation>> entry : cryptographicSuite.getAcceptableSignatureAlgorithms().entrySet()) {
            SignatureAlgorithm signatureAlgorithm = entry.getKey();
            Set<CryptographicSuiteEvaluation> evaluations = entry.getValue();
            Integer minKeyLength = null;
            if (Utils.isCollectionNotEmpty(evaluations)) {
                for (CryptographicSuiteEvaluation evaluation : evaluations) {
                    if (!CryptographicSuiteUtils.isEvaluationApplicable(signatureAlgorithm.getEncryptionAlgorithm(), null, evaluation)) continue;
                    Date validityStart = evaluation.getValidityStart();
                    Date validityEnd = evaluation.getValidityEnd();
                    if (validityStart != null && !validityStart.before(validationTime) || validityEnd != null && !validityEnd.after(validationTime)) continue;
                    int keyLength = CryptographicSuiteUtils.getMinKeyLength(signatureAlgorithm.getEncryptionAlgorithm(), evaluation);
                    if (minKeyLength != null && minKeyLength <= keyLength) continue;
                    minKeyLength = keyLength;
                }
            }
            if (minKeyLength == null) continue;
            result.add(new SignatureAlgorithmWithMinKeySize(signatureAlgorithm, minKeyLength));
        }
        return result;
    }

    public static boolean isSignatureAlgorithmReliableAtTime(CryptographicSuite cryptographicSuite, SignatureAlgorithm signatureAlgorithm, String keyLength, Date validationTime) {
        Set<CryptographicSuiteEvaluation> evaluations = cryptographicSuite.getAcceptableSignatureAlgorithms().get(signatureAlgorithm);
        return CryptographicSuiteUtils.reliableEvaluationExistsAtTime(signatureAlgorithm.getEncryptionAlgorithm(), CryptographicSuiteUtils.parseKeySize(keyLength), evaluations, validationTime);
    }

    private static boolean isEvaluationApplicable(EncryptionAlgorithm algorithm, Integer keySize, CryptographicSuiteEvaluation evaluation) {
        List<CryptographicSuiteParameter> parameterList = evaluation.getParameterList();
        if (parameterList == null || parameterList.isEmpty()) {
            return true;
        }
        for (CryptographicSuiteParameter parameter : parameterList) {
            if (!CryptographicSuiteUtils.isSupported(algorithm, parameter) || keySize != null && (parameter.getMin() != null && keySize < parameter.getMin() || parameter.getMax() != null && keySize > parameter.getMax())) continue;
            return true;
        }
        return false;
    }

    private static int getMinKeyLength(EncryptionAlgorithm algorithm, CryptographicSuiteEvaluation evaluation) {
        Integer minKeyLength = null;
        List<CryptographicSuiteParameter> parameterList = evaluation.getParameterList();
        if (parameterList != null && !parameterList.isEmpty()) {
            for (CryptographicSuiteParameter parameter : parameterList) {
                if (!CryptographicSuiteUtils.isSupported(algorithm, parameter)) continue;
                if (parameter.getMin() == null) {
                    return 0;
                }
                if (minKeyLength != null && minKeyLength <= parameter.getMin()) continue;
                minKeyLength = parameter.getMin();
            }
        }
        return minKeyLength != null ? minKeyLength : 0;
    }

    private static boolean isSupported(EncryptionAlgorithm encryptionAlgorithm, CryptographicSuiteParameter parameter) {
        String parameterName = parameter.getName();
        if (MODULES_LENGTH_PARAMETER.equals(parameterName)) {
            if (EncryptionAlgorithm.RSA.isEquivalent(encryptionAlgorithm)) {
                return true;
            }
        } else if (PLENGTH_PARAMETER.equals(parameterName)) {
            if (EncryptionAlgorithm.DSA.isEquivalent(encryptionAlgorithm) || EncryptionAlgorithm.ECDSA.isEquivalent(encryptionAlgorithm) || EncryptionAlgorithm.EDDSA.isEquivalent(encryptionAlgorithm)) {
                return true;
            }
        } else if (!QLENGTH_PARAMETER.equals(parameterName)) {
            LOG.debug("Unknown Algorithms Parameter type '{}'!", (Object)parameterName);
        }
        return false;
    }
}

