/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.signature;

import eu.europa.esig.dss.enumerations.SigningOperation;
import eu.europa.esig.dss.model.signature.SignatureCryptographicVerification;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.signature.AbstractSignatureParameters;
import eu.europa.esig.dss.spi.exception.IllegalInputException;
import eu.europa.esig.dss.spi.signature.AdvancedSignature;
import eu.europa.esig.dss.spi.validation.CertificateVerifier;
import eu.europa.esig.dss.spi.validation.CertificateVerifierBuilder;
import eu.europa.esig.dss.spi.validation.RevocationDataVerifier;
import eu.europa.esig.dss.spi.validation.SignatureValidationAlerter;
import eu.europa.esig.dss.spi.validation.SignatureValidationContext;
import eu.europa.esig.dss.spi.validation.TimestampTokenVerifier;
import eu.europa.esig.dss.spi.validation.ValidationAlerter;
import eu.europa.esig.dss.spi.validation.status.SignatureStatus;
import eu.europa.esig.dss.utils.Utils;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SignatureRequirementsChecker {
    private static final Logger LOG = LoggerFactory.getLogger(SignatureRequirementsChecker.class);
    protected final CertificateVerifier certificateVerifier;
    protected final AbstractSignatureParameters<?> signatureParameters;

    public SignatureRequirementsChecker(CertificateVerifier certificateVerifier, AbstractSignatureParameters<?> signatureParameters) {
        Objects.requireNonNull(certificateVerifier, "CertificateVerifier cannot be null!");
        Objects.requireNonNull(signatureParameters, "Signature parameters cannot be null!");
        this.certificateVerifier = certificateVerifier;
        this.signatureParameters = signatureParameters;
    }

    public void assertSigningCertificateIsValid(CertificateToken certificateToken) {
        ValidationAlerter validationAlerter = this.initValidationAlerter(certificateToken);
        this.assertCertificatesAreYetValid(validationAlerter, certificateToken);
        this.assertCertificatesAreNotExpired(validationAlerter, certificateToken);
        this.assertCertificatesAreNotRevoked(validationAlerter, certificateToken);
    }

    public void assertSigningCertificateIsValid(Collection<AdvancedSignature> signatures) {
        List signaturesToValidate = signatures.stream().filter(s -> !this.isSignatureGeneratedWithoutCertificate((AdvancedSignature)s)).collect(Collectors.toList());
        if (Utils.isCollectionEmpty(signaturesToValidate)) {
            return;
        }
        ValidationAlerter validationAlerter = this.initValidationAlerter(signatures);
        this.assertCertificatesAreYetValid(validationAlerter);
        this.assertCertificatesAreNotExpired(validationAlerter);
        this.assertCertificatesAreNotRevoked(validationAlerter);
    }

    private boolean isSignatureGeneratedWithoutCertificate(AdvancedSignature signature) {
        if (this.isSigningCertificateIdentified(signature)) {
            return false;
        }
        if (this.signatureParameters.isGenerateTBSWithoutCertificate()) {
            LOG.debug("Signature with Id '{}' has been generated without certificate or signing-certificate has not been identified. Validity of the signing-certificate is not checked.", (Object)signature.getId());
            return true;
        }
        throw new IllegalInputException("Signing-certificate token was not found! Unable to verify its validity. Provide signing-certificate or use method #setGenerateTBSWithoutCertificate(true) for signature creation without signing-certificate.");
    }

    private boolean isSigningCertificateIdentified(AdvancedSignature signature) {
        return signature.getCertificateSource().getNumberOfCertificates() != 0 && signature.getSigningCertificateToken() != null;
    }

    private void assertCertificatesAreYetValid(ValidationAlerter validationAlerter) {
        this.assertCertificatesAreYetValid(validationAlerter, null);
    }

    private void assertCertificatesAreYetValid(ValidationAlerter validationAlerter, CertificateToken certificateToken) {
        if (this.certificateVerifier.getAlertOnNotYetValidCertificate() == null) {
            LOG.trace("The verification of #certificatesAreYetValid has been skipped.");
            return;
        }
        if (certificateToken != null) {
            validationAlerter.assertCertificateIsYetValid(certificateToken);
        } else {
            validationAlerter.assertAllSignaturesAreYetValid();
        }
    }

    private void assertCertificatesAreNotExpired(ValidationAlerter validationAlerter) {
        this.assertCertificatesAreNotExpired(validationAlerter, null);
    }

    private void assertCertificatesAreNotExpired(ValidationAlerter validationAlerter, CertificateToken certificateToken) {
        if (this.certificateVerifier.getAlertOnExpiredCertificate() == null) {
            LOG.trace("The verification of #certificatesAreNotExpired has been skipped.");
            return;
        }
        if (certificateToken != null) {
            validationAlerter.assertCertificateNotExpired(certificateToken);
        } else {
            validationAlerter.assertAllSignaturesNotExpired();
        }
    }

    private void assertCertificatesAreNotRevoked(ValidationAlerter validationAlerter) {
        this.assertCertificatesAreNotRevoked(validationAlerter, null);
    }

    private void assertCertificatesAreNotRevoked(ValidationAlerter validationAlerter, CertificateToken certificateToken) {
        if (!this.signatureParameters.isCheckCertificateRevocation()) {
            return;
        }
        if (this.certificateVerifier.getAlertOnMissingRevocationData() == null && this.certificateVerifier.getAlertOnRevokedCertificate() == null) {
            LOG.trace("The verification of #certificatesAreNotRevoked has been skipped.");
            return;
        }
        validationAlerter.assertAllRequiredRevocationDataPresent();
        if (certificateToken != null) {
            validationAlerter.assertCertificateNotRevoked(certificateToken);
        } else {
            validationAlerter.assertAllSignatureCertificatesNotRevoked();
        }
    }

    protected ValidationAlerter initValidationAlerter(CertificateToken certificateToken) {
        SignatureValidationContext validationContext = new SignatureValidationContext(this.signatureParameters.bLevel().getSigningDate());
        validationContext.initialize(this.getCertificateVerifier());
        List<CertificateToken> certificateChain = this.signatureParameters.getCertificateChain();
        if (Utils.isCollectionEmpty(certificateChain)) {
            if (this.signatureParameters.isCheckCertificateRevocation()) {
                throw new NullPointerException("Certificate chain shall be provided for a revocation check! Please use parameters.setCertificateChain(...) method to provide a certificate chain.");
            }
            certificateChain = Collections.emptyList();
        }
        validationContext.addCertificateTokenForVerification(certificateToken);
        for (CertificateToken certificate : certificateChain) {
            validationContext.addCertificateTokenForVerification(certificate);
        }
        validationContext.validate();
        SignatureValidationAlerter signatureValidationAlerter = new SignatureValidationAlerter(validationContext);
        signatureValidationAlerter.setSigningOperation(SigningOperation.SIGN);
        return signatureValidationAlerter;
    }

    protected ValidationAlerter initValidationAlerter(Collection<AdvancedSignature> signatures) {
        SignatureValidationContext validationContext = new SignatureValidationContext(this.signatureParameters.bLevel().getSigningDate());
        validationContext.initialize(this.getCertificateVerifier());
        for (AdvancedSignature signature : signatures) {
            validationContext.addSignatureForVerification(signature);
        }
        validationContext.validate();
        SignatureValidationAlerter signatureValidationAlerter = new SignatureValidationAlerter(validationContext);
        signatureValidationAlerter.setSigningOperation(SigningOperation.EXTEND);
        return signatureValidationAlerter;
    }

    protected CertificateVerifier getCertificateVerifier() {
        if (this.signatureParameters.isCheckCertificateRevocation()) {
            return this.certificateVerifier;
        }
        CertificateVerifier offlineCertificateVerifier = new CertificateVerifierBuilder(this.certificateVerifier).buildOfflineCopy();
        RevocationDataVerifier acceptAllRevocationDataVerifier = this.createAcceptAllRevocationDataVerifier();
        offlineCertificateVerifier.setRevocationDataVerifier(acceptAllRevocationDataVerifier);
        TimestampTokenVerifier timestampTokenVerifier = offlineCertificateVerifier.getTimestampTokenVerifier();
        if (timestampTokenVerifier == null) {
            timestampTokenVerifier = TimestampTokenVerifier.createDefaultTimestampTokenVerifier();
        }
        timestampTokenVerifier.setRevocationDataVerifier(acceptAllRevocationDataVerifier);
        return offlineCertificateVerifier;
    }

    private RevocationDataVerifier createAcceptAllRevocationDataVerifier() {
        RevocationDataVerifier revocationDataVerifier = RevocationDataVerifier.createDefaultRevocationDataVerifier();
        revocationDataVerifier.setAcceptRevocationCertificatesWithoutRevocation(true);
        revocationDataVerifier.setAcceptTimestampCertificatesWithoutRevocation(true);
        return revocationDataVerifier;
    }

    public void assertExtendToTLevelPossible(List<AdvancedSignature> signatures) {
        this.assertTLevelIsHighest(signatures);
        this.assertHasNoEmbeddedEvidenceRecords(signatures);
    }

    protected void assertTLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #tLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkTLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to T-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert(status);
        }
    }

    protected void checkTLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasLTLevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasLTLevelOrHigher(AdvancedSignature signature) {
        return signature.hasLTAProfile() || (signature.hasLTProfile() || signature.hasCProfile()) && !signature.areAllSelfSignedCertificates() && signature.hasTProfile();
    }

    public void assertExtendToLTLevelPossible(List<AdvancedSignature> signatures) {
        this.assertLTLevelIsHighest(signatures);
        this.assertHasNoEmbeddedEvidenceRecords(signatures);
    }

    protected void assertLTLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #ltLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkLTLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to LT-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert(status);
        }
    }

    protected void checkLTLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasLTALevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasLTALevelOrHigher(AdvancedSignature signature) {
        return signature.hasLTAProfile();
    }

    public void assertCertificateChainValidForLTLevel(List<AdvancedSignature> signatures) {
        this.assertCertificateChainValid(signatures, "LT");
    }

    public void assertCertificateChainValidForCLevel(List<AdvancedSignature> signatures) {
        this.assertCertificateChainValid(signatures, "C");
    }

    public void assertCertificateChainValidForXLLevel(List<AdvancedSignature> signatures) {
        this.assertCertificateChainValid(signatures, "XL");
    }

    private void assertCertificateChainValid(List<AdvancedSignature> signatures, String targetLevel) {
        this.assertCertificatePresent(signatures, targetLevel);
        this.assertCertificatesAreNotSelfSigned(signatures, targetLevel);
    }

    private void assertCertificatePresent(List<AdvancedSignature> signatures, String targetLevel) {
        if (this.certificateVerifier.getAugmentationAlertOnSignatureWithoutCertificates() == null) {
            LOG.trace("The verification of #certificatePresent has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            if (signature.getCertificateSource().getNumberOfCertificates() != 0) continue;
            status.addRelatedTokenAndErrorMessage(signature, "The signature does not contain certificates.");
        }
        if (!status.isEmpty()) {
            status.setMessage(String.format("Error on signature augmentation to %s-level.", targetLevel));
            this.certificateVerifier.getAugmentationAlertOnSignatureWithoutCertificates().alert(status);
        }
    }

    private void assertCertificatesAreNotSelfSigned(List<AdvancedSignature> signatures, String targetLevel) {
        if (this.certificateVerifier.getAugmentationAlertOnSelfSignedCertificateChains() == null) {
            LOG.trace("The verification of #certificatesAreNotSelfSigned has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            if (!signature.areAllSelfSignedCertificates()) continue;
            status.addRelatedTokenAndErrorMessage(signature, "The signature contains only self-signed certificate chains.");
        }
        if (!status.isEmpty()) {
            status.setMessage(String.format("Error on signature augmentation to %s-level.", targetLevel));
            this.certificateVerifier.getAugmentationAlertOnSelfSignedCertificateChains().alert(status);
        }
    }

    public void assertExtendToCLevelPossible(List<AdvancedSignature> signatures) {
        this.assertCLevelIsHighest(signatures);
        this.assertHasNoEmbeddedEvidenceRecords(signatures);
    }

    protected void assertCLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #cLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkCLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to C-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert(status);
        }
    }

    protected void checkCLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasXLevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasXLevelOrHigher(AdvancedSignature signature) {
        return signature.hasXProfile() || signature.hasAProfile() || signature.hasXLProfile() && !signature.areAllSelfSignedCertificates() && signature.hasTProfile();
    }

    public void assertExtendToXLevelPossible(List<AdvancedSignature> signatures) {
        this.assertXLevelIsHighest(signatures);
        this.assertHasNoEmbeddedEvidenceRecords(signatures);
    }

    protected void assertXLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #xLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkXLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to X-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert(status);
        }
    }

    protected void checkXLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasXLLevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasXLLevelOrHigher(AdvancedSignature signature) {
        return signature.hasAProfile() || signature.hasXLProfile() && !signature.areAllSelfSignedCertificates() && signature.hasTProfile() && signature.hasXProfile();
    }

    public void assertExtendToXLLevelPossible(List<AdvancedSignature> signatures) {
        this.assertXLLevelIsHighest(signatures);
        this.assertHasNoEmbeddedEvidenceRecords(signatures);
    }

    protected void assertXLLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #xlLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkXLLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to XL-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert(status);
        }
    }

    protected void checkXLLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasALevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasALevelOrHigher(AdvancedSignature signature) {
        return this.hasLTALevelOrHigher(signature);
    }

    public void assertExtendToLTALevelPossible(List<AdvancedSignature> signatures) {
        this.assertHasNoEmbeddedEvidenceRecords(signatures);
    }

    protected void assertHasNoEmbeddedEvidenceRecords(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #hasEmbeddedEvidenceRecords has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkHasEmbeddedEvidenceRecords(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert(status);
        }
    }

    protected void checkHasEmbeddedEvidenceRecords(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasEmbeddedEvidenceRecords(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is preserved by an embedded evidence record.");
        }
    }

    public boolean hasEmbeddedEvidenceRecords(AdvancedSignature signature) {
        return Utils.isCollectionNotEmpty(signature.getEmbeddedEvidenceRecords());
    }

    public void assertSignaturesValid(Collection<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAlertOnInvalidSignature() == null) {
            LOG.trace("The verification of #signaturesValid has been skipped.");
            return;
        }
        List signaturesToValidate = signatures.stream().filter(s -> !this.isSignatureGeneratedWithoutCertificate((AdvancedSignature)s)).collect(Collectors.toList());
        if (Utils.isCollectionEmpty(signaturesToValidate)) {
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signaturesToValidate) {
            SignatureCryptographicVerification signatureCryptographicVerification = signature.getSignatureCryptographicVerification();
            if (signatureCryptographicVerification.isSignatureValid()) continue;
            String errorMessage = signatureCryptographicVerification.getErrorMessage();
            status.addRelatedTokenAndErrorMessage(signature, "Cryptographic signature verification has failed" + (errorMessage.isEmpty() ? "." : " / " + errorMessage));
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation");
            this.certificateVerifier.getAlertOnInvalidSignature().alert(status);
        }
    }
}

